﻿# Script to apply WorkSpace CSBs to folders within a Work Area based on name.

# WorkSpace Configuration Blocks to set.
#     Global            - Global Variables
#     Workset/Project   - MyNewWorkArea
#     Workset/Project   - MyNewWorkArea - Architectural 
#     Workset/Project   - MyNewWorkArea - Civil
#     Organization/Site - MyCompany
#     Predefined        - MicroStation
#     Predefined        - OBD
#     Predefined        - ORD    

Get-Help about_Switch

# WorkArea folders.                  
$arFolders = @(
    'Projects\MyNewWorkArea'
    'Projects\MyNewWorkArea\Architectural'
    'Projects\MyNewWorkArea\Civil'
    'Projects\MyNewWorkArea\Electrical'
    'Projects\MyNewWorkArea\General'
    'Projects\MyNewWorkArea\Geotechnical'
    'Projects\MyNewWorkArea\HVAC'
    'Projects\MyNewWorkArea\Instrumentation'
    'Projects\MyNewWorkArea\Landscape'
    'Projects\MyNewWorkArea\Mech-proc'
    'Projects\MyNewWorkArea\Plumbing and Fire Protection'
    'Projects\MyNewWorkArea\Structural'
    'Projects\MyNewWorkArea\Architectural\Model'
    'Projects\MyNewWorkArea\Structural\Model'
    'Projects\MyNewWorkArea\Civil\Model'
    'Projects\MyNewWorkArea\Electrical\P&ID'
    'Projects\MyNewWorkArea\Instrumentation\Model'
) # end $arFolders...

# Create new folders to be used in script.
foreach ($f in $arfolders){
    $newPWFolder = New-PWFolder -FolderPath "$f" -Description "$f Discipline" -StorageArea 'Storage01' -Verbose 
} # end foreach ($f in $arfolders...
# Convert parent folder to Work Area.
Convert-PWFolderToRichProject -InputFolder (Get-PWFolders -FolderPath $arFolders[0] -JustOne)
    
# Get all folders pertaining to the desired Work Area.
$pwFolders = Get-PWFolders -FolderPath 'Projects\MyNewWorkArea' -PopulatePaths

foreach($pwFolder in $pwFolders) {

    Write-Host "Current folder '$($pwFolder.FullPath)'." -ForegroundColor Cyan
    
    # Capturing name and fullpath to simplify.
    $currentName = $pwFolder.Name
    $currentFullPath = $pwFolder.FullPath
    
    $arFolderNames = @('Architectural', 'civil', 'electrical', 'general', 'geotechnical', 'hvac', 'instrumentation', 'landscape', 'mech-proc', 'plumbing and fire protection', 'structural')
    
    switch ($currentName) {
        
        # Determines if the current folder name is in the array of folder names.
        # Set WorkSet CSBs here.
        { $_ -in $arFolderNames } {
            Write-Host "$currentName." -ForegroundColor DarkMagenta
            Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType WorkSet -ConfigurationBlockName 'MyNewWorkArea'}" -ForegroundColor Yellow
            break
        } # end { $_ -in $arFolderNames...
             
        # Set WorkSet and User for Instrumentation folder ONLY.
        'instrumentation' {
            Write-Host "$currentName" -ForegroundColor Green
            Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType WorkSet -ConfigurationBlockName 'MyNewWorkArea'}" -ForegroundColor Yellow
            Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType User    -ConfigurationBlockName 'Instrumentation'}" -ForegroundColor Yellow
            break
        } # end 'instrumentation'...
        
        # Set for Model or P&ID folders ONLY.
        { $_ -in @('model', 'P&ID') } {
            Write-Host "$currentName" -ForegroundColor Green
            
            # Set OBD (Open Building Designer) CSB for Architectural folders.
            if($currentFullPath.Contains('architectural')) {
                Write-Host " - Architectural Model" -ForegroundColor DarkGreen
                Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType Predefined -ConfigurationBlockName 'OBD'}" -ForegroundColor Yellow
            }

            # Set ORD (Open Roads Designer) CSB for Civil folders.
            if($currentFullPath.Contains('civil')) {
                Write-Host " - Civil Model" -ForegroundColor DarkGreen
                Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType Predefined -ConfigurationBlockName 'ORD'}" -ForegroundColor Yellow
            }
   
            if($currentFullPath.Contains('electrical')) {
                Write-Host " - Electrical Model" -ForegroundColor DarkGree
            }
              
            # Set MicroStation CSB for all folders.
            Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType Predefined -ConfigurationBlockName 'MicroStation'}" -ForegroundColor Yellow          
            break
        } # end {$_ -in @('model', 'P&ID')...
   
        # Default should ONLY apply to the Work Area folder.
        #    Set Global configuration ONLY.           
        default {
            Write-Host "Global" -ForegroundColor Magenta
            Write-Host " - Set Global Workspace CSBs Here." -ForegroundColor Magenta
            Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType Global -ConfigurationBlockName 'Global Variables'}" -ForegroundColor Yellow
            Write-Host " - {Set-PWManagedWorkspaceForFolders -InputFolders $pwFolder -WorkspaceType Site   -ConfigurationBlockName 'MyCompany'}" -ForegroundColor Yellow
            break
        } # end default...
                    
    } # end switch ($currentName...
    
} # end foreach($pwFolder in $pwFolders...
          


# Cleanup Work Area created for script.
Remove-PWFolder -InputFolder (Get-PWFolders -FolderPath $arFolders[0] -JustOne) -RemoveFolders -ProceedWithDelete -Verbose


